import os
import datetime
import argparse
from tixi3.tixi3wrapper import Tixi3
from tigl3.tigl3wrapper import Tigl3

from myTool.myTool import MyDummyTool


class ToolWrapper:
    def __init__(self, cpacs_input, cpacs_output):
        self.cpacs_input = cpacs_input
        self.cpacs_output = cpacs_output

        self.tixi = Tixi3()
        self.tigl = Tigl3()

        self.model_uid = None

        self.tool_data = {
            "a": None,
            "b": None,
        }

    def preprocessing(self):

        # Load data with TiXI:
        try:
            self.tixi.openDocument(self.cpacs_input)
        except Exception as e:
            raise RuntimeError(f"TiXI could not load {self.cpacs_input}! Error: {e}")

        # Select aircraft model for further evaluation with TiGL:
        self.model_uid = self.tixi.getTextAttribute(
            "/cpacs/vehicles/aircraft/model[1]", "uID"
        )

        try:
            self.tigl.open(self.tixi, self.model_uid)
        except Exception as e:
            raise RuntimeError(f"Could not load TiGL! Error: {e}")

        # =======================================
        #   Read data from CPAS
        # =======================================
        # This involves reading the data from CPACS (e.g. using Tixi, etree, or other XML parsers) and
        # preparing it for the wrapped tool. Of course, this is very much dependent on the form in which
        # the tool to be wrapped accepts data. Is it written to an extra input file? Should the data be
        # transferred to a json dictionary? A wide variety of approaches are conceivable and can be
        # implemented here.

        # Example: Reading Operating Empty Mass with TiXI

        # model_xpath = self.tixi.uIDGetXPath(self.model_uid)
        # massBreakdown_xpath = model_xpath + "/analyses/massBreakdown"
        # self.tool_data["a"] = self.tixi.getDoubleElement(
        #     massBreakdown_xpath + "/mOEM/massDescription/mass"
        # )

    def compute(self):
        # =======================================
        #   Call your actual tool
        # =======================================
        # You have to be creative at this point. Depending on the tool architecture,
        # there might be different ways to call it. For example:
        #     - Direct Python Library Call (see below)
        #     - Subprocess-Based Execution (see Python’s subprocess module)
        #     - Web API or Service Integration (see Python’s request module)
        #     - ...

        try:
            my_dummy_tool = MyDummyTool(self.tool_data)
            my_dummy_tool.run()
            self.tool_data = my_dummy_tool.get_data()
        except Exception as e:
            raise RuntimeError(f"Error while executing the tool: {e}")

    def postprocessing(self):

        # =======================================
        #   Write tool data back to CPACS
        # =======================================
        # This is the part where the data generated by your individual tool is transferred to
        # the CPACS data structure and written to the XML file. By default, each tool should add
        # a short info in the header node to ensure traceability. Specific CPACS data is then
        # written to the <vehicle> or other CPACS nodes.

        # (1) Add header info:
        version = self.tixi.getTextElement("/cpacs/header/version")
        version_info_xpath = (
            f"/cpacs/header/versionInfos/versionInfo[@version='{version}']"
        )

        change_log_xpath = version_info_xpath + "/changeLog"
        if not self.tixi.checkElement(change_log_xpath):
            self.tixi.createElement(version_info_xpath, "changeLog")

        log_entry_xpath = change_log_xpath + "/logEntry[last()]"
        self.tixi.createElement(change_log_xpath, "logEntry")
        self.tixi.addTextElement(
            log_entry_xpath, "description", "Add my dummy tool data"
        )

        timestamp = datetime.datetime.now().strftime("%Y-%m-%dT%H:%M:%S")
        self.tixi.addTextElement(log_entry_xpath, "timestamp", timestamp)
        self.tixi.addTextElement(log_entry_xpath, "creator", "Dummy Tool")

        # (2) Writing tool data back to CPACS, for example aerodynamics:

        # model_xpath = self.tixi.uIDGetXPath(self.model_uid)
        # self._addElement(
        #     model_xpath + "/analyses/aeroPerformance/aeroMap[1]/aeroPerformanceMap/cl",
        #     "%.3f" % self.tool_data["results"]["cl"],
        # )
        # self._addElement(
        #     model_xpath + "/analyses/aeroPerformance/aeroMap[1]/aeroPerformanceMap/cd",
        #     "%.3f" % self.tool_data["results"]["cd"],
        # )

        # (3) Save output to XML:
        if self.cpacs_output:
            output_dir = os.path.dirname(self.cpacs_output)
            if output_dir:
                os.makedirs(output_dir, exist_ok=True)
            self.tixi.save(self.cpacs_output)
            self.tixi.close()

    def _addElement(self, xpath, value=None, overwrite=True):
        """Help function to add string elements with TiXI. Not required, but handy."""
        elements = xpath.strip("/").split("/")
        current_xpath = ""

        for i, elem in enumerate(elements):
            current_xpath += f"/{elem}"

            if i == len(elements) - 1 and value is not None:
                if self.tixi.checkElement(current_xpath):
                    if overwrite:
                        self.tixi.updateTextElement(current_xpath, value)
                    else:
                        print(
                            f"Warnung: Element {current_xpath} exists and will not be overwritten."
                        )
                else:
                    self.tixi.addTextElement(
                        "/".join(elements[:-1]), elements[-1], value
                    )

            elif not self.tixi.checkElement(current_xpath):
                clean_elem = elem.split("[")[0]
                self.tixi.createElement("/".join(elements[:i]), clean_elem)

    def run(self):
        print("========= Structures =========")
        self.preprocessing()
        self.compute()
        self.postprocessing()


if __name__ == "__main__":

    parser = argparse.ArgumentParser(description="My dummy tool wrapper")

    parser.add_argument(
        "--cpacs_input",
        type=str,
        default=os.path.join(os.getcwd(), "cpacsIO", "cpacs_in.xml"),
        help="Path to the CPACS input file",
    )

    parser.add_argument(
        "--cpacs_output",
        type=str,
        default=os.path.join(os.getcwd(), "cpacsIO", "cpacs_out.xml"),
        help="Path to the CPACS output file",
    )

    args = parser.parse_args()

    tool = ToolWrapper(
        cpacs_input=args.cpacs_input, cpacs_output=args.cpacs_output
    )
    tool.run()
